/*
 * Decompiled with CFR 0.152.
 */
package ru.quadcom.database.lib.cassandra.utils;

import com.datastax.driver.core.DataType;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.core.schemabuilder.Create;
import com.datastax.driver.core.schemabuilder.CreateType;
import com.datastax.driver.core.schemabuilder.SchemaBuilder;
import com.datastax.driver.core.schemabuilder.SchemaStatement;
import com.datastax.driver.mapping.annotations.ClusteringColumn;
import com.datastax.driver.mapping.annotations.Column;
import com.datastax.driver.mapping.annotations.Defaults;
import com.datastax.driver.mapping.annotations.PartitionKey;
import com.datastax.driver.mapping.annotations.Table;
import com.datastax.driver.mapping.annotations.UDT;
import com.google.common.base.CaseFormat;
import com.google.common.base.Predicate;
import com.stratio.cassandra.lucene.builder.Builder;
import com.stratio.cassandra.lucene.builder.index.Index;
import com.stratio.cassandra.lucene.builder.index.schema.mapping.Mapper;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import org.reflections.ReflectionUtils;
import org.reflections.Reflections;
import org.reflections.scanners.Scanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.quadcom.database.lib.cassandra.annotations.LuceneField;
import ru.quadcom.database.lib.cassandra.annotations.LuceneIndex;
import ru.quadcom.database.lib.cassandra.annotations.Secondary;
import ru.quadcom.database.lib.cassandra.exceptions.TableAnnotationNotFoundCassandraRuntimeException;
import ru.quadcom.database.lib.cassandra.exceptions.UnsupportedCassandraIndexTypeException;
import ru.quadcom.database.lib.cassandra.impl.CassandraClient;

public class CassandraUtils {
    private static final Logger logger = LoggerFactory.getLogger(CassandraUtils.class);
    private static Reflections reflect;
    private static Map<Class<?>, UDT> customTypes;
    private static Map<Class<?>, Table> tables;

    public static void InitTables(CassandraClient client, String searchPath) {
        reflect = new Reflections(searchPath, new Scanner[0]);
        customTypes.clear();
        for (Class clazz : reflect.getTypesAnnotatedWith(UDT.class)) {
            UDT udt = clazz.getAnnotation(UDT.class);
            customTypes.put(clazz, udt);
        }
        tables.clear();
        for (Class clazz : reflect.getTypesAnnotatedWith(Table.class)) {
            Table table = clazz.getAnnotation(Table.class);
            tables.put(clazz, table);
        }
        if (tables.size() == 0) {
            logger.warn("CassandraUtils: no table beans found in " + searchPath);
            return;
        }
        client.getSession().thenComposeAsync(session -> {
            ArrayList<CompletableFuture<Void>> tasks = new ArrayList<CompletableFuture<Void>>();
            for (Class<?> type : customTypes.keySet()) {
                logger.debug("Checking custom type: " + type.toString());
                tasks.add(CassandraUtils.checkAndCreateCustomType(client, session, type));
            }
            for (Class<?> table : tables.keySet()) {
                logger.debug("Checking table: " + table.toString());
                Field[] keyFields = CassandraUtils.getPartitionKeys(table);
                if (keyFields.length == 0) continue;
                tasks.add(CassandraUtils.checkAndCreateTable(client, session, table, keyFields));
            }
            return tasks.size() > 0 ? CompletableFuture.allOf(tasks.toArray(new CompletableFuture[tasks.size()])) : CompletableFuture.completedFuture(null);
        }).exceptionally(problem -> {
            logger.error("CassandraUtils initTable Error: ", problem);
            return null;
        });
    }

    public static String getTableName(Class table) {
        if (!tables.containsKey(table)) {
            throw new TableAnnotationNotFoundCassandraRuntimeException("No annotation @CassandraTable present in " + table.toString());
        }
        return tables.get(table).name();
    }

    public static String getIndexName(Class table) {
        if (!table.isAnnotationPresent(LuceneIndex.class)) {
            throw new TableAnnotationNotFoundCassandraRuntimeException("No annotation @LuceneIndex present in " + table.toString());
        }
        LuceneIndex ci = table.getAnnotation(LuceneIndex.class);
        return ci.name();
    }

    private static CompletableFuture<Void> checkAndCreateCustomType(CassandraClient client, Session session, Class type) {
        String typeName = customTypes.get(type).name();
        logger.debug("Type name: " + typeName);
        return CompletableFuture.supplyAsync(() -> {
            logger.debug("Creating UDT " + typeName);
            String createDDL = CassandraUtils.generateTypeDDL(type, typeName);
            logger.info("Running DDL: " + createDDL);
            session.execute(createDDL);
            return null;
        });
    }

    private static CompletableFuture<Void> checkAndCreateTable(CassandraClient client, Session session, Class table, Field[] partitionKeys) {
        String tableName = CassandraUtils.getTableName(table);
        Field[] clusteringKeys = CassandraUtils.getClusteringKeys(table);
        logger.debug("Table name: " + tableName + " primary keys count: " + partitionKeys.length + " clustering keys count: " + clusteringKeys.length);
        return client.isTableExists(tableName).thenAcceptAsync(exists -> {
            if (!exists.booleanValue()) {
                logger.debug("Creating table " + tableName + " exists");
                String createDDL = CassandraUtils.generateDDL(table, tableName);
                logger.info("Running DDL: " + createDDL);
                session.execute(createDDL);
            } else {
                logger.debug("Table " + tableName + " exists");
            }
        }).thenComposeAsync(ignore -> {
            String luceneIndexName = CassandraUtils.getLuceneIndex(table);
            if (luceneIndexName != null) {
                return client.getIndexOptions(tableName, luceneIndexName).thenApplyAsync(options -> {
                    String indexDDL = CassandraUtils.makeLuceneIndexDDL(table, tableName, luceneIndexName);
                    if (options == null) {
                        logger.info("Running lucene index DDL: " + indexDDL);
                        session.execute(indexDDL);
                    } else {
                        logger.debug("Table " + tableName + " index " + luceneIndexName + " exists");
                    }
                    return null;
                });
            }
            return CompletableFuture.completedFuture(null);
        }).thenAcceptAsync(ignore -> {
            Map<String, String> secondary_indexes = CassandraUtils.getSecondaryIndexes(table);
            if (secondary_indexes != null && secondary_indexes.size() > 0) {
                for (Map.Entry<String, String> index : secondary_indexes.entrySet()) {
                    SchemaStatement builder = SchemaBuilder.createIndex((String)index.getKey()).ifNotExists().onTable(tableName).andColumn(index.getValue());
                    logger.info("Running secondary index DDL: " + builder.getQueryString());
                    session.execute((Statement)builder);
                }
            }
        }).toCompletableFuture();
    }

    private static String makeLuceneIndexDDL(Class table, String tableName, String indexName) {
        LuceneIndex index = table.getAnnotation(LuceneIndex.class);
        Index indexBuilder = Builder.index((String)tableName, (String)indexName).refreshSeconds((Number)index.refreshSeconds());
        for (Field field : ReflectionUtils.getAllFields((Class)table, (Predicate[])new Predicate[]{ReflectionUtils.withAnnotation(LuceneField.class)})) {
            indexBuilder.mapper(CassandraUtils.toCassandraColumnName(field.getName()), CassandraUtils.getIndexTypeMapper(field.getType()));
        }
        return indexBuilder.toString();
    }

    private static String getLuceneIndex(Class table) {
        if (!table.isAnnotationPresent(LuceneIndex.class)) {
            return null;
        }
        LuceneIndex index = table.getAnnotation(LuceneIndex.class);
        return index.name();
    }

    private static Map<String, String> getSecondaryIndexes(Class table) {
        HashMap<String, String> names = new HashMap<String, String>();
        for (Field field : ReflectionUtils.getAllFields((Class)table, (Predicate[])new Predicate[]{ReflectionUtils.withAnnotation(Secondary.class)})) {
            names.put(field.getAnnotation(Secondary.class).indexName(), CassandraUtils.getSecondaryKeyName(field));
        }
        return names;
    }

    private static Field[] getPartitionKeys(Class table) {
        Set primary = ReflectionUtils.getAllFields((Class)table, (Predicate[])new Predicate[]{ReflectionUtils.withAnnotation(PartitionKey.class)});
        if (primary.size() == 0) {
            logger.error("CassandraUtils: Found zero partition keys in " + table.toString());
            return new Field[0];
        }
        Field[] keys = new Field[primary.size()];
        return primary.toArray(keys);
    }

    private static Field[] getClusteringKeys(Class table) {
        Set primary = ReflectionUtils.getAllFields((Class)table, (Predicate[])new Predicate[]{ReflectionUtils.withAnnotation(ClusteringColumn.class)});
        Field[] keys = new Field[primary.size()];
        return primary.toArray(keys);
    }

    private static String getColumnName(Field field) {
        Column p = field.getAnnotation(Column.class);
        return CassandraUtils.toCassandraColumnName(p.name().equals("") ? field.getName() : p.name());
    }

    private static String getSecondaryKeyName(Field field) {
        Secondary s = field.getAnnotation(Secondary.class);
        return CassandraUtils.toCassandraColumnName(s.name().equals("") ? field.getName() : s.name());
    }

    private static String generateTypeDDL(Class typeClass, String typeName) {
        CreateType create = (CreateType)SchemaBuilder.createType((String)typeName).ifNotExists();
        for (Field field : typeClass.getDeclaredFields()) {
            if (Modifier.isStatic(field.getModifiers())) continue;
            Column column = field.getAnnotation(Column.class);
            if (column != null && column.codec() != Defaults.NoCodec.class) {
                try {
                    Class clazz = column.codec();
                    Constructor ctor = clazz.getDeclaredConstructor(new Class[0]);
                    create.addColumn(CassandraUtils.toCassandraColumnName(field.getName()), ((TypeCodec)ctor.newInstance(new Object[0])).getCqlType());
                }
                catch (Exception e) {
                    logger.error("Error getting type codec from Column in UDT class: " + typeClass);
                    create.addColumn(CassandraUtils.toCassandraColumnName(field.getName()), CassandraUtils.getCassandraType(field.getType()));
                }
                continue;
            }
            create.addColumn(CassandraUtils.toCassandraColumnName(field.getName()), CassandraUtils.getCassandraType(field.getType()));
        }
        return create.getQueryString();
    }

    private static String generateDDL(Class table, String tableName) {
        Create create = (Create)SchemaBuilder.createTable((String)tableName).ifNotExists();
        for (Field field : table.getDeclaredFields()) {
            DataType type;
            if (Modifier.isStatic(field.getModifiers())) continue;
            Column column = field.getAnnotation(Column.class);
            if (column != null && column.codec() != Defaults.NoCodec.class) {
                try {
                    Class clazz = column.codec();
                    Constructor ctor = clazz.getDeclaredConstructor(new Class[0]);
                    type = ((TypeCodec)ctor.newInstance(new Object[0])).getCqlType();
                }
                catch (Exception e) {
                    logger.error("Error getting type codec from Column in Table class: " + table);
                    type = CassandraUtils.getCassandraType(field.getType());
                }
            } else {
                type = CassandraUtils.getCassandraType(field.getType());
            }
            if (field.isAnnotationPresent(PartitionKey.class)) {
                create.addPartitionKey(CassandraUtils.getColumnName(field), type);
                continue;
            }
            if (field.isAnnotationPresent(ClusteringColumn.class)) {
                create.addClusteringColumn(CassandraUtils.getColumnName(field), type);
                continue;
            }
            if (customTypes.containsKey(type)) {
                String udtName = customTypes.get(type).name();
                create.addUDTColumn(CassandraUtils.toCassandraColumnName(field.getName()), SchemaBuilder.frozen((String)udtName));
                continue;
            }
            create.addColumn(CassandraUtils.toCassandraColumnName(field.getName()), type);
        }
        return create.getQueryString();
    }

    private static String getCassandraTypeStr(Class type) {
        if (Integer.class.isAssignableFrom(type) || Integer.TYPE.isAssignableFrom(type)) {
            return "int";
        }
        if (Long.class.isAssignableFrom(type) || Long.TYPE.isAssignableFrom(type)) {
            return "bigint";
        }
        if (Float.class.isAssignableFrom(type) || Float.TYPE.isAssignableFrom(type)) {
            return "float";
        }
        if (Boolean.class.isAssignableFrom(type) || Boolean.TYPE.isAssignableFrom(type)) {
            return "boolean";
        }
        if (UUID.class.isAssignableFrom(type)) {
            return "uuid";
        }
        if (List.class.isAssignableFrom(type)) {
            return "list";
        }
        if (Properties.class.isAssignableFrom(type) || Map.class.isAssignableFrom(type)) {
            return "map<varchar,varchar>";
        }
        if (String.class.isAssignableFrom(type)) {
            return "varchar";
        }
        if (Date.class.isAssignableFrom(type)) {
            return "varchar";
        }
        return "varchar";
    }

    private static DataType getCassandraType(Class type) {
        if (Integer.class.isAssignableFrom(type) || Integer.TYPE.isAssignableFrom(type)) {
            return DataType.cint();
        }
        if (Long.class.isAssignableFrom(type) || Long.TYPE.isAssignableFrom(type)) {
            return DataType.bigint();
        }
        if (Float.class.isAssignableFrom(type) || Float.TYPE.isAssignableFrom(type)) {
            return DataType.cfloat();
        }
        if (Boolean.class.isAssignableFrom(type) || Boolean.TYPE.isAssignableFrom(type)) {
            return DataType.cboolean();
        }
        if (UUID.class.isAssignableFrom(type)) {
            return DataType.uuid();
        }
        if (String.class.isAssignableFrom(type)) {
            return DataType.varchar();
        }
        if (Properties.class.isAssignableFrom(type) || Map.class.isAssignableFrom(type)) {
            return DataType.map((DataType)DataType.varchar(), (DataType)DataType.varchar());
        }
        if (List.class.isAssignableFrom(type)) {
            return DataType.list((DataType)DataType.varchar());
        }
        if (Date.class.isAssignableFrom(type)) {
            return DataType.varchar();
        }
        throw new UnsupportedCassandraIndexTypeException("Unknown index type: " + type.toString());
    }

    private static Mapper getIndexTypeMapper(Class type) {
        if (Integer.class.isAssignableFrom(type) || Integer.TYPE.isAssignableFrom(type)) {
            return Builder.integerMapper();
        }
        if (Long.class.isAssignableFrom(type) || Long.TYPE.isAssignableFrom(type)) {
            return Builder.bigIntegerMapper();
        }
        if (Float.class.isAssignableFrom(type) || Float.TYPE.isAssignableFrom(type)) {
            return Builder.floatMapper();
        }
        if (String.class.isAssignableFrom(type)) {
            return Builder.stringMapper();
        }
        if (UUID.class.isAssignableFrom(type)) {
            return Builder.uuidMapper();
        }
        if (Boolean.class.isAssignableFrom(type) || Boolean.TYPE.isAssignableFrom(type)) {
            return Builder.booleanMapper();
        }
        if (Properties.class.isAssignableFrom(type) || Map.class.isAssignableFrom(type)) {
            throw new UnsupportedCassandraIndexTypeException("Map or Properties classes not supported");
        }
        if (Date.class.isAssignableFrom(type)) {
            return Builder.stringMapper();
        }
        throw new UnsupportedCassandraIndexTypeException("Unknown index type: " + type.toString());
    }

    public static String toCassandraColumnName(String javaField) {
        return CaseFormat.LOWER_CAMEL.to(CaseFormat.LOWER_UNDERSCORE, javaField);
    }

    private static String fromCassandraColumnName(String cassandraField) {
        return CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, cassandraField);
    }

    static {
        customTypes = new HashMap();
        tables = new HashMap();
    }
}

