/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cloud.aws.blobstore;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import com.amazonaws.util.Base64;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import org.elasticsearch.cloud.aws.blobstore.S3BlobStore;
import org.elasticsearch.cloud.aws.blobstore.S3OutputStream;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;

public class DefaultS3OutputStream
extends S3OutputStream {
    private static final ByteSizeValue MULTIPART_MAX_SIZE = new ByteSizeValue(5L, ByteSizeUnit.GB);
    private static final ESLogger logger = Loggers.getLogger((String)"cloud.aws");
    private String multipartId;
    private int multipartChunks;
    private List<PartETag> multiparts;

    public DefaultS3OutputStream(S3BlobStore blobStore, String bucketName, String blobName, int bufferSizeInBytes, int numberOfRetries, boolean serverSideEncryption) {
        super(blobStore, bucketName, blobName, bufferSizeInBytes, numberOfRetries, serverSideEncryption);
    }

    @Override
    public void flush(byte[] bytes, int off, int len, boolean closing) throws IOException {
        if ((long)len > MULTIPART_MAX_SIZE.getBytes()) {
            throw new IOException("Unable to upload files larger than " + MULTIPART_MAX_SIZE + " to Amazon S3");
        }
        if (!closing) {
            if (len < this.getBufferSize()) {
                this.upload(bytes, off, len);
            } else {
                if (this.getFlushCount() == 0) {
                    this.initializeMultipart();
                }
                this.uploadMultipart(bytes, off, len, false);
            }
        } else if (this.multipartId != null) {
            this.uploadMultipart(bytes, off, len, true);
            this.completeMultipart();
        } else {
            this.upload(bytes, off, len);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void upload(byte[] bytes, int off, int len) throws IOException {
        try (ByteArrayInputStream is = new ByteArrayInputStream(bytes, off, len);){
            for (int retry = 0; retry <= this.getNumberOfRetries(); ++retry) {
                try {
                    this.doUpload(this.getBlobStore(), this.getBucketName(), this.getBlobName(), is, len, this.isServerSideEncryption());
                    return;
                }
                catch (AmazonClientException e) {
                    if (!this.getBlobStore().shouldRetry(e) || retry >= this.getNumberOfRetries()) throw new IOException("Unable to upload object " + this.getBlobName() + " due to " + ((Object)((Object)e)).getClass().getSimpleName() + ": " + e.getMessage());
                    is.reset();
                    continue;
                    return;
                }
            }
        }
    }

    protected void doUpload(S3BlobStore blobStore, String bucketName, String blobName, InputStream is, int length, boolean serverSideEncryption) throws AmazonS3Exception {
        MessageDigest messageDigest;
        ObjectMetadata md = new ObjectMetadata();
        if (serverSideEncryption) {
            md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
        }
        md.setContentLength((long)length);
        InputStream inputStream = is;
        try {
            messageDigest = MessageDigest.getInstance("MD5");
            inputStream = new DigestInputStream(is, messageDigest);
        }
        catch (NoSuchAlgorithmException impossible) {
            throw new RuntimeException(impossible);
        }
        PutObjectResult putObjectResult = blobStore.client().putObject(bucketName, blobName, inputStream, md);
        String localMd5 = Base64.encodeAsString((byte[])messageDigest.digest());
        String remoteMd5 = putObjectResult.getContentMd5();
        if (!localMd5.equals(remoteMd5)) {
            logger.debug("MD5 local [{}], remote [{}] are not equal...", new Object[]{localMd5, remoteMd5});
            throw new AmazonS3Exception("MD5 local [" + localMd5 + "], remote [" + remoteMd5 + "] are not equal...");
        }
    }

    private void initializeMultipart() throws IOException {
        int retry = 0;
        while (retry <= this.getNumberOfRetries() && this.multipartId == null) {
            try {
                this.multipartId = this.doInitialize(this.getBlobStore(), this.getBucketName(), this.getBlobName(), this.isServerSideEncryption());
                if (this.multipartId == null) continue;
                this.multipartChunks = 1;
                this.multiparts = new ArrayList<PartETag>();
            }
            catch (AmazonClientException e) {
                if (this.getBlobStore().shouldRetry(e) && retry < this.getNumberOfRetries()) {
                    ++retry;
                    continue;
                }
                throw new IOException("Unable to initialize multipart request for object " + this.getBlobName() + " due to " + ((Object)((Object)e)).getClass().getSimpleName() + ": " + e.getMessage());
            }
        }
    }

    protected String doInitialize(S3BlobStore blobStore, String bucketName, String blobName, boolean serverSideEncryption) {
        InitiateMultipartUploadRequest request = new InitiateMultipartUploadRequest(bucketName, blobName);
        if (serverSideEncryption) {
            ObjectMetadata md = new ObjectMetadata();
            md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
            request.setObjectMetadata(md);
        }
        return blobStore.client().initiateMultipartUpload(request).getUploadId();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void uploadMultipart(byte[] bytes, int off, int len, boolean lastPart) throws IOException {
        Throwable throwable = null;
        try (ByteArrayInputStream is = new ByteArrayInputStream(bytes, off, len);){
            for (int retry = 0; retry <= this.getNumberOfRetries(); ++retry) {
                try {
                    PartETag partETag = this.doUploadMultipart(this.getBlobStore(), this.getBucketName(), this.getBlobName(), this.multipartId, is, len, lastPart);
                    this.multiparts.add(partETag);
                    ++this.multipartChunks;
                    return;
                }
                catch (AmazonClientException e) {
                    try {
                        if (this.getBlobStore().shouldRetry(e) && retry < this.getNumberOfRetries()) {
                            is.reset();
                            continue;
                        }
                        this.abortMultipart();
                        throw new IOException("Unable to upload multipart request [" + this.multipartId + "] for object " + this.getBlobName() + " due to " + ((Object)((Object)e)).getClass().getSimpleName() + ": " + e.getMessage());
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    catch (Throwable throwable3) {
                        throw throwable3;
                        return;
                    }
                }
            }
        }
    }

    protected PartETag doUploadMultipart(S3BlobStore blobStore, String bucketName, String blobName, String uploadId, InputStream is, int length, boolean lastPart) throws AmazonS3Exception {
        UploadPartRequest request = new UploadPartRequest().withBucketName(bucketName).withKey(blobName).withUploadId(uploadId).withPartNumber(this.multipartChunks).withInputStream(is).withPartSize((long)length).withLastPart(lastPart);
        UploadPartResult response = blobStore.client().uploadPart(request);
        return response.getPartETag();
    }

    private void completeMultipart() throws IOException {
        for (int retry = 0; retry <= this.getNumberOfRetries(); ++retry) {
            try {
                this.doCompleteMultipart(this.getBlobStore(), this.getBucketName(), this.getBlobName(), this.multipartId, this.multiparts);
                this.multipartId = null;
                return;
            }
            catch (AmazonClientException e) {
                if (this.getBlobStore().shouldRetry(e) && retry < this.getNumberOfRetries()) {
                    continue;
                }
                this.abortMultipart();
                throw new IOException("Unable to complete multipart request [" + this.multipartId + "] for object " + this.getBlobName() + " due to " + ((Object)((Object)e)).getClass().getSimpleName() + ": " + e.getMessage());
            }
        }
    }

    protected void doCompleteMultipart(S3BlobStore blobStore, String bucketName, String blobName, String uploadId, List<PartETag> parts) throws AmazonS3Exception {
        CompleteMultipartUploadRequest request = new CompleteMultipartUploadRequest(bucketName, blobName, uploadId, parts);
        blobStore.client().completeMultipartUpload(request);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void abortMultipart() {
        if (this.multipartId != null) {
            try {
                this.doAbortMultipart(this.getBlobStore(), this.getBucketName(), this.getBlobName(), this.multipartId);
            }
            finally {
                this.multipartId = null;
            }
        }
    }

    protected void doAbortMultipart(S3BlobStore blobStore, String bucketName, String blobName, String uploadId) throws AmazonS3Exception {
        blobStore.client().abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, blobName, uploadId));
    }
}

